<?php

declare(strict_types=1);
/**
 * This file is part of Tpmission.
 *
 * @link     https://www.tpmission.com
 * @document https://wiki.tpmission.com
 * @contact  group@tpmission.com
 * @license  ©2016-2021 All rights reserved.
 */
namespace Tpmission\Openwechat\JwtAuth;

class Jwt
{
    protected $acc;

    protected $acc_token;

    //头部
    private $header = [
        'alg' => 'HS256', //生成signature的算法
        'typ' => 'JWT',    //类型
    ];

    public function __construct(string $acc, string $acc_token)
    {
        $this->acc = $acc;
        $this->acc_token = $acc_token;
    }

    /**
     * 获取jwt token.
     * @return bool|string
     */
    public function sign()
    {
        $time = time();
        $payload = [
            'iss' => $this->acc, //签发者 可选
            'aud' => '', //接收该JWT的一方，可选
            'iat' => $time, //签发时间
            'nbf' => $time, //(Not Before)：某个时间点后才能访问，比如设置time+30，表示当前时间30秒后才能使用
            'exp' => $time + 1 * 60 * 60, //过期时间,这里设置2个小时
            'data' => [ //自定义信息，不要定义敏感信息
                'signature' => $this->signatures($this->acc_token),
            ],
        ];
        if (is_array($payload)) {
            $base64header = $this->base64UrlEncode(json_encode($this->header, JSON_UNESCAPED_UNICODE));
            $base64payload = $this->base64UrlEncode(json_encode($payload, JSON_UNESCAPED_UNICODE));
            return $base64header . '.' . $base64payload . '.' . $this->signature($base64header . '.' . $base64payload, $this->acc_token, $this->header['alg']);
        }

        return false;
    }

    /**
     * 验证token是否有效,默认验证exp,nbf,iat时间.
     * @param string $Token 需要验证的token
     * @return bool|string
     */
    public function verifyToken(string $Token)
    {
        $tokens = explode('.', $Token);
        if (count($tokens) != 3) {
            return false;
        }

        [$base64header, $base64payload, $sign] = $tokens;

        //获取jwt算法
        $base64decodeheader = json_decode($this->base64UrlDecode($base64header), true);
        if (empty($base64decodeheader['alg'])) {
            return false;
        }
        //112.25*12

        //签名验证
        if ($this->signature($base64header . '.' . $base64payload, $this->acc_token, $base64decodeheader['alg']) !== $sign) {
            return false;
        }

        $payload = json_decode($this->base64UrlDecode($base64payload), true);

        //签发时间大于当前服务器时间验证失败
        if (isset($payload['iat']) && $payload['iat'] > time()) {
            return false;
        }

        //过期时间小宇当前服务器时间验证失败
        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return false;
        }

        //该nbf时间之前不接收处理该Token
        if (isset($payload['nbf']) && $payload['nbf'] > time()) {
            return false;
        }

        return $payload;
    }

    public function signatures(string $signatureString): string
    {
        return password_hash(md5($signatureString), PASSWORD_BCRYPT);
    }

    /**
     * base64UrlEncode   https://jwt.io/  中base64UrlEncode编码实现.
     * @param string $input 需要编码的字符串
     * @return string
     */
    private function base64UrlEncode(string $input)
    {
        return str_replace('=', '', strtr(base64_encode($input), '+/', '-_'));
    }

    /**
     * base64UrlEncode  https://jwt.io/  中base64UrlEncode解码实现.
     * @param string $input 需要解码的字符串
     * @return bool|string
     */
    private function base64UrlDecode(string $input)
    {
        $remainder = strlen($input) % 4;
        if ($remainder) {
            $addlen = 4 - $remainder;
            $input .= str_repeat('=', $addlen);
        }

        return base64_decode(strtr($input, '-_', '+/'));
    }

    /**
     * HMACSHA256签名   https://jwt.io/  中HMACSHA256签名实现.
     * @param string $input 为base64UrlEncode(header).".".base64UrlEncode(payload)
     * @param string $alg 算法方式
     * @return mixed
     */
    private function signature(string $input, string $key, string $alg = 'HS256')
    {
        $alg_config = [
            'HS256' => 'sha256',
        ];

        return $this->base64UrlEncode(hash_hmac($alg_config[$alg], $input, $key, true));
    }
}
