<?php

declare(strict_types=1);
/**
 * This file is part of Tpmission.
 *
 * @link     https://www.tpmission.com
 * @document https://wiki.tpmission.com
 * @contact  group@tpmission.com
 * @license  https://github.com/hyperf/hyperf/blob/master/LICENSE
 */
namespace Tpmission\Openwechat\Common;

use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\TransferException;
use Tpmission\Openwechat\Common\Http\HttpClient;
use Tpmission\Openwechat\JwtAuth\Jwt;

class BaseClient
{
    protected $guzzleOptions = [];

    protected $client;

    protected $config = [];

    //protected $base_uri="http://127.0.0.1:9501/api/";
    protected $base_uri = 'http://open-wechat-service/api/';

    protected $api = '';

    protected $get_token_api = 'apiAuth';

    protected $refresh_token_api = 'authRefresh';

    protected $http_method = 'post';

    protected $jwt = '';

    protected $msg_types = [
        'news', 'text', 'img', 'voice', 'video',
    ];

    protected $qrcodeTypes = [
        'forever', 'temporary',
    ];

    public function __construct(SdkContainer $sdkContainer)
    {
        $this->config = $sdkContainer->getConfig();
        $this->init_token();
        $httpClient = new HttpClient();
        $httpClient->setGuzzleOptions([
            'headers' => [
                'Authorization' => $this->jwt,
                'Author' => $this->config['acc'],
            ],
        ]);
        $this->client = $httpClient->getHttpClient();
    }

    public function make_return($response, $response_type = '')
    {
        $response_type = $response_type ? $response_type : ($this->config['response_type'] ?? '');
        switch ($response_type) {
            case 'json':
                return $response;
                break;
            case 'xml':
                $xml_data = json_decode($response, true);
                return $this->xml_encode($xml_data, 'utf-8', 'response');
                break;
            case 'array':
                return \json_decode($response, true);
                break;
            default:
                return $response;
                break;
        }
    }

    /**
     * XML编码
     * @param mixed $data 数据
     * @param string $encoding 数据编码
     * @param string $root 根节点名
     * @return string
     */
    public function xml_encode($data, $encoding = 'utf-8', $root = 'Response')
    {
        $xml = '<?xml version="1.0" encoding="utf-8"?>';
        $xml .= '<' . $root . '>';
        $xml .= $this->data_to_xml($data);
        $xml .= '</' . $root . '>';
        return $xml;
    }

    /**
     * 数据XML编码
     * @param mixed $data 数据
     * @return string
     */
    public function data_to_xml($data)
    {
        $xml = '';
        foreach ($data as $key => $val) {
            is_numeric($key) && $key = "item id=\"{$key}\"";
            $xml .= "<{$key}>";
            $xml .= (is_array($val) || is_object($val)) ? $this->data_to_xml($val) : $val;
            [$key, ] = explode(' ', $key);
            $xml .= "</{$key}>";
        }
        return $xml;
    }

    protected function init($nameSpace)
    {
        $api = str_replace('Tpmission\Openwechat\\', '', $nameSpace);
        $api = str_replace('\\', '', $api);
        $this->api = lcfirst($api);
    }

    /**
     *token初始化.
     */
    protected function init_token()
    {
        $jwt = new Jwt($this->config['acc'], $this->config['token']);
        $jwt = $jwt->sign();
        $this->jwt = $jwt;
        if (! $this->jwt) {
            throw new \Exception('获取接口调用凭证失败');
        }
    }

    protected function http_run(array $parm)
    {
        $parm['appid'] = $this->config['appid'];
        $url = $this->base_uri . $this->api;
        $http_method = $this->http_method;
        try {
            $response = $this->client->{$http_method}($url, [
                'query' => $parm,
            ]);
            return $this->make_return($response->getBody()->getContents());
        } catch (TransferException $e) {
            $message = $e->getMessage();
            if ($e instanceof ConnectException) { //异常发生在网络错误
                $err = '连接接口失败，接口访问服务异常。';
                $josn = json_encode([
                    'code' => 10400,
                    'msg' => $err,
                    'data' => [
                        'errorCode' => 9001,
                        'errorMsg' => $message,
                        'data' => [$e->getCode(), $url],
                    ],
                ]);
                return $this->make_return($josn);
            }
            if ($e instanceof RequestException) {
                $response = $e->getResponse();
                $code = $response->getStatusCode(); // 200
                $res = $response->getBody()->getContents();
                if (! json_decode($res, true)) {
                    $res = json_encode(
                        [
                            'code' => '10' . $code,
                            'msg' => '连接接口成功，接口处理请求失败',
                            'data' => ['errorCode' => 9000, 'errorMsg' => $res, 'data' => []],
                        ]
                    );
                }
                return $this->make_return($res);
            }
        }
    }

    protected function Errs($msg, $code = 0)
    {
        throw new \Tpmission\Openwechat\Exceptions\InvalidArgumentException($msg, $code);
    }
}
